//*************************************************************************************************
//
//	Description:
//		debug.fx - A shader which allows the user to query various non-visible properties of a
//							 rendered object.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     06/30/2006  0.1           Created
//		TNettleship     10/16/2006  0.2           Changed the way lighting distance attenuation works.
//		TNettleship			10/19/2006	0.3						Removed texture mapping info params from display,
//																							reverted to parser 0x0000 until the texture management
//																							bugs in 0x0001 can be fixed.
//		TNettleship			11/02/2006	0.4						Fixed a bug with directional lighting.
//		TMann						12/06/2006	0.5						Win_GL version
//		TMann						12/11/2006	0.6						PS3 version
//		TNettleship			10/23/2007	0.7						Converted to on-load rendermode behaviour binding.
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"

//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

// Lighting
#ifdef _3DSMAX_
#define MAX_LIGHTS 1
#else
#define MAX_LIGHTS 1
#endif

#define LIGHTTYPE_NONE 0
#define LIGHTTYPE_AMBIENT 1
#define LIGHTTYPE_DIRECTIONAL 2
#define LIGHTTYPE_SPOTLIGHT 3
#define LIGHTTYPE_POINT 4

#if defined(_PS3_)
#define _MINFILTER	LinearMipMapLinear
#else
#define _MINFILTER	Linear
#endif


//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//

#if defined( _3DSMAX_ )
// Max doesn't support viewproj as an app-supplied parameter
SHARE_PARAM float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;
float4x4 worldI : WorldInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;


//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

#endif


//
// Display options
//

bool normalRendering
<
	string UIName = "Normal rendering?";
> = true;

bool showDepth
<
	string UIName = "Show depth?";
> = false;

float depthScale
<
	string UIName = "Depth scale";
> = 100.0f;

bool showVertexColour
<
	string UIName = "Show vertex colour?";
> = false;

bool showNormal
<
	string UIName = "Show normal?";
> = false;

bool showTangent
<
	string UIName = "Show tangent?";
> = false;

bool showBinormal
<
	string UIName = "Show binormal?";
> = false;

bool showAlpha
<
	string UIName = "Show alpha?";
> = false;

bool showUVMapping
<
	string UIName = "Show UV mapping?";
> = false;

bool showSpecularPower
<
	string UIName = "Show specular power?";
> = false;

bool showNormalLength
<
	string UIName = "Show normal length?";
> = false;

bool showTangentLength
<
	string UIName = "Show tangent length?";
> = false;

bool showBinormalLength
<
	string UIName = "Show binormal length?";
> = false;

bool showUVGradients
<
	string UIName = "Show UV gradients?";
> = false;

//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Texture";
	bool appEdit = true;
>;

texture specularTexture : TEXTURE							// Specular colour in RGB, shininess in alpha
<
	string UIName = "Specular Texture";
	bool appEdit = true;
>;

float specularRollOff													// Specular lighting overlighting rolloff amount
<
	string UIName = "Specular Rolloff";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 0.1f;

bool useNormalMap															// TRUE if the normal map is to be used in lighting
<
	string UIName = "Use normal map?";
	bool appEdit = true;
> = false;

texture normalTexture : TEXTURE								// Normal map + specularity factor in alpha.
<
	string UIName = "Normal Texture";
	bool appEdit = true;
>;

bool specialNormalMap													// Use compressed / platform-specific normal map format?
<
	string UIName = "Special normal map format?";
	bool appEdit = true;
> = false;

bool useEnvironmentMap												// TRUE if the environment map is to be used
<
	string UIName = "Use environment map?";
	bool appEdit = true;
> = false;

texture environmentTexture : TEXTURE					// Environment map (RGB)
<
	string UIName = "Env Texture";
	string Type = "CUBE";
	bool appEdit = false;
	bool export = false;
>;


//
// Lighting
//
float minSpecPower
<
	string UIName = "Min Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 1.0f;

float maxSpecPower
<
	string UIName = "Max Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 32.0f;

float globalSpecularFactor
<
	string UIName = "Specular Factor";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;

float globalEMapFactor
<
	string UIName = "EMap Factor";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;

float4 ambientColour : Ambient								// Ambient colour
<
	string UIName = "Ambient Colour";
	string UIType = "ColorSwatch";
	bool appEdit = true;
> = {0.25f, 0.25f, 0.25f, 0.0f};

#ifdef _3DSMAX_

float lightPower															// Light power level.
<
	string UIName = "Light Power";
	string UIType = "FloatSpinner";
	float UIMin = 0.0f;
	float UIMax = 100.0f;
	bool appEdit = false;
	bool export = false;
> = 1.0f;																			

int lightType														// Light type.
<
	string UIName = "Light type";
	string UIType = "IntSpinner";
	int UIMin = 0;
	int UIMax = 4;
	bool appEdit = false;
	bool export = false;
> = LIGHTTYPE_DIRECTIONAL;

float innerAngle
<
	string UIName = "Spotlight Inner Angle";
	float UIMin = 0.0f;
	float UIMax = 90.0f;
	bool appEdit = false;
	bool export = false;
> = 10.0f;

float outerAngle
<
	string UIName = "Spotlight Outer Angle";
	float UIMin = 0.0f;
	float UIMax = 90.0f;
	bool appEdit = false;
	bool export = false;
> = 12.0f;

float3 lightPosition : POSITION								// Light position, world coords
<
	string UIName = "Light Position";
	string Object = "PointLight";
	string Space = "World";
	int RefID = 0;
	bool appEdit = false;
	bool export = false;
>;

float3 lightDirection : Direction						// Light direction, world coords
<
	string UIName = "Light Direction";
	string Object = "SpotLight";
	string Space = "World";
	int LightRef = 0;
	bool appEdit = false;
	bool export = false;
>;

float4 lightColour : LightColor							// Light colour.
<
	string UIWidget = "None";
	int LightRef = 0;
	bool appEdit = false;
	bool export = false;
> = {1.0f, 1.0f, 1.0f, 1.0f};

float lightAttenStop
<
	string UIName = "Light Atten Stop";
	string UIType = "FloatSpinner";
	float UIMin = 0.0f;
	float UIMax = 100000.0f;
	bool appEdit = false;
	bool export = false;
> = 100000.0f;

#else

int lightType[ MAX_LIGHTS ] : lightType										// Light types.
<
	bool appEdit = false;
>;

float lightIntensity[ MAX_LIGHTS ] : lightIntensity				// Light intensity levels. First level at 0.0 == end of the lights list.
<
	bool appEdit = false;
>;

float4 lightColour[ MAX_LIGHTS ] : lightColour						// Light colours.
<
	bool appEdit = false;
>;

float3 lightDirection[ MAX_LIGHTS ] : lightDirection			// Light directions, world coords (for directional and spot lights).
<
	bool appEdit = false;
>;

float3 lightPosition[ MAX_LIGHTS ] : lightPosition				// Light positions, world coords
<
	bool appEdit = false;
>;

float lightAttenuation[ MAX_LIGHTS ] : lightAttenuation		// Distance attenuation factor: x: 1 / ( end dist squared ).
<
	bool appEdit = false;
>;

float2 lightAngle[ MAX_LIGHTS ] : lightAngle							// Spolight inner/outer angle factors (x: inner, y: outer).
<
	bool appEdit = false;
>;

bool shadowing[ MAX_LIGHTS ] : lightShadow								// TRUE if light source casts shadows, FALSE if not.
<
	bool appEdit = false;
>;

#endif



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER < bool appEdit = false; > = sampler_state
{
	FX_SAMPLERSTATE_SRGB_TEXTURE
	Texture = < diffuseTexture >;
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
};

sampler2D specularMap : SAMPLER < bool appEdit = false; > = sampler_state
{
	FX_SAMPLERSTATE_SRGB_TEXTURE
	Texture = < specularTexture >;
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
};

sampler2D normalMap : SAMPLER < bool appEdit = false; > = sampler_state
{
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	Texture = < normalTexture >;
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
};

samplerCUBE environmentMap : SAMPLER < bool appEdit = false; > = sampler_state
{
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	Texture = < environmentTexture >;
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
};


//-----------------------------------------------------------------------
//
// Functions
//

//-----------------------------------------------------------------------
//
// CalculateAttenuation - Calculates the attenuation factor for a directional, point
//												or spot lightsource at a given point.
//
// Parameters:
//	_pointToLight : Vector from the point to the light source.
//	_dir					: Light direction (relevant for directional and spot light source.)
//	_attFactor		: Attenuation factors ( = 1.0 / ( end dist squared )
//	_type					: Type of light source (0 = none, 1 = directional, 2 = spotlight, 3 = point light)
//	_angles				: Cosines of the inner (x) and outer (y) spotlight angles.
//
// Return:
//	The attenuation factor for the light source at this position, [0.0 ... 1.0].
//
//-----------------------------------------------------------------------

float CalculateLightAttenuation( float3 _pointToLight, float3 _dir, float _attFactor, int _type, float2 _angles )
{
	float att;

	// Directional light
	if ( _type == LIGHTTYPE_DIRECTIONAL )
	{
		att = 1.0f;
	}

	else
	{
		// Point light
		if ( _type == LIGHTTYPE_POINT )
		{
				float distSquared = dot( _pointToLight, _pointToLight );
				att = smoothstep( 0.0f, 1.0f, 1.0f - ( distSquared * _attFactor ) );
		}

		// Spot light
		else
		{
			// Calculate distance attenuation
			float distSquared = dot( _pointToLight, _pointToLight );
			att = smoothstep( 0.0f, 1.0f, 1.0f - ( distSquared * _attFactor ) );

			// Calculate angle attenuation
			float cosAngle = - dot( _dir, _pointToLight / sqrt( distSquared ) );
			float angleAttenuation = smoothstep( _angles.y, _angles.x, cosAngle );

			// Combine the two
			att *= angleAttenuation;
		}
	}

	return att;
}



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float4 colour   : TEXCOORD1;													// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
#endif
	float3 normal   : NORMAL;															// Object space normal
	float3 tangent  : TANGENT;														// Object space tangent
	float3 binormal : BINORMAL;														// Object space normal
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: COLOR0;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float3 tangent		: TEXCOORD1;												// Tangent vector (object space)
	float3 normal			: TEXCOORD2;												// Normal vector (object space)
	float3 binormal		: TEXCOORD3;												// Normal vector (object space)
	float3 eye				: TEXCOORD4;												// Eye vector (object space)
	float3 worldPos		: TEXCOORD5;												// World-coords position
	float3 depth			: TEXCOORD6;												// Depth
#ifdef _3DSMAX_
	float3 lightVector : TEXCOORD7;												// Light vector (object space)
#else
	float3 lightVector[ MAX_LIGHTS ] : TEXCOORD7;					// Light vectors (object space)
#endif
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//
// TODO: rearrange to transform eye and lights into local space before
// vector calculations.

VSOUTPUT DebugVertexShader( VSINPUT _input )
{
#if defined(_PS3_)
	VSOUTPUT _output;
#else
	VSOUTPUT _output = ( VSOUTPUT )0;
#endif	

#if !defined( _3DSMAX_ )
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.colour = _input.colour;
	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );
	_output.depth = (float3)_output.position;

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;
	_output.worldPos = worldPos;

	// Calculate object-space coordinate frame
	_output.tangent  = _input.tangent;
	_output.normal   = _input.normal;
	_output.binormal  = _input.binormal;

	// Calculate object-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = mul( float4( worldEyeVec, 0.0f ), worldI ).xyz;

#ifdef _3DSMAX_
	if ( lightType == LIGHTTYPE_DIRECTIONAL )
	{
		float3 worldLightVec = lightDirection;
		_output.lightVector = mul( float4( worldLightVec, 0.0f ), worldI ).xyz;
	}
	else
	{
		// Calculate object-space vector to the light source
		float3 worldLightVec = lightPosition - worldPos;
		_output.lightVector = mul( float4( worldLightVec, 0.0f ), worldI ).xyz;
	}
#else
	// For each light source
	for ( int index = 0; index < MAX_LIGHTS; index++ )
	{
		if ( lightType[ index ] == LIGHTTYPE_DIRECTIONAL )
		{
			float3 worldLightVec = - lightDirection[ index ];
			_output.lightVector[ index ] = mul( float4( worldLightVec, 0.0f ), worldI ).xyz;
		}
		else
		{
			// Calculate object-space vector to the light source
			float3 worldLightVec = lightPosition[ index ] - worldPos;
			_output.lightVector[ index ] = mul( float4( worldLightVec, 0.0f ), worldI ).xyz;
		}
	}
#endif

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Input structure
struct PSINPUT
{
	float4 colour			: COLOR0;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float3 tangent		: TEXCOORD1;												// Tangent vector (object space)
	float3 normal			: TEXCOORD2;												// Normal vector (object space)
	float3 binormal		: TEXCOORD3;												// Normal vector (object space)
	float3 eye				: TEXCOORD4;												// Eye vector (object space)
	float3 worldPos		: TEXCOORD5;												// World-coords position
	float3 depth			: TEXCOORD6;												// Depth
#ifdef _3DSMAX_
	float3 lightVector : TEXCOORD7;												// Light vector (object space)
#else
	float3 lightVector[ MAX_LIGHTS ] : TEXCOORD7;					// Light vectors (object space)
#endif
};


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT DebugFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output = ( PSOUTPUT )0;

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );
	float4 specularTexColour = tex2D( specularMap, _input.texCoord );

	// If normal shading is required
	if ( normalRendering )
	{
		// Normalise interpolated vectors
		float3 TSnormal = normalize( _input.normal );
		float3 tangent = normalize( _input.tangent );
		float3 binormal = normalize( _input.binormal );
		float3 eye = normalize( _input.eye );
		float3 normal;

		// If normal map support is required
		if ( useNormalMap )
		{
			// Fetch and decode the map normal
			float4 normalMapColour = tex2D( normalMap, _input.texCoord );
			float3 normalFromMap;

			if ( specialNormalMap )
			{
				normalFromMap.rgb = normalize( ( normalMapColour.agb * 2.0f ) - 1.0f );
			}
			else
			{
				normalFromMap.rgb = normalize( ( normalMapColour.rgb * 2.0f ) - 1.0f );
			}

			// Perturb the tangent space normal by the normal map
			normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
			normal = normalize( normal );
		}

		// No normal map, so use interpolated normal and constant specular strength
		else
		{
			normal = TSnormal;
		}

	//	float4 accumulatedColour = (float4)0.0f;
		// Calculate ambient colour
		float4 accumulatedColour = ambientColour * diffuseTexColour * _input.colour;

		// If environment mapping is switched on
		if ( useEnvironmentMap )
		{
			// Fetch the environment map colour
			float3 reflectionVector = reflect( -eye, normal );
#ifdef _3DSMAX_
			float4 environmentTexColour = texCUBE( environmentMap, reflectionVector.xzy );
#else
			float4 environmentTexColour = texCUBE( environmentMap, reflectionVector );
#endif

			// Add this to the accumulated lighting (treat it as specular light)
			float specularPower = lerp( minSpecPower, maxSpecPower, specularTexColour.a );
			if ( ( specularPower > 0.0f ) && ( globalEMapFactor > 0.0f ) )
			{
				// Calculate envmap colour and add to diffuse
				accumulatedColour += specularTexColour * CalculateSpecularLighting( normal, reflectionVector, eye, specularPower ) * environmentTexColour * globalEMapFactor;
			}
		}

#ifdef _3DSMAX_
		// If the light source is switched on
		if ( lightPower > 0.0f )
		{
			// Calculate light attenuation
			float3 lightDir = mul( float4( lightDirection, 0.0f ), worldI ).xyz;
			float2 angles = float2( cos( radians( innerAngle * 0.5f ) ), cos( radians( outerAngle * 0.5f ) ) );
			float attFactor;
			attFactor = 1.0f / ( lightAttenStop * lightAttenStop );
			float attenuation = CalculateLightAttenuation( _input.lightVector,
																										 -lightDir,
																										 attFactor,
																										 lightType,
																										 angles );

			if ( attenuation > 0.0f )
			{
				// Normalise the vector to the light source
				float3 lightVector = normalize( _input.lightVector );

				// Calculate diffuse colour and add to accumulator
				float diffuseLevel = CalculateDiffuseLighting( normal, lightVector );
				accumulatedColour += diffuseTexColour * max( diffuseLevel, 0.0f ) * lightColour * lightPower * attenuation;

				// If some specular lighting should be applied
				float specularPower = lerp( minSpecPower, maxSpecPower, specularTexColour.a );
				if ( ( specularPower > 0.0f ) && ( globalSpecularFactor > 0.0f ) )
				{
					// If the fragment faces the light source
					if ( diffuseLevel > 0.0f )
					{
						// Calculate specular colour and add to diffuse
						accumulatedColour += specularTexColour * CalculateSpecularLighting( normal, lightVector, eye, specularPower ) * lightColour * lightPower * globalSpecularFactor * attenuation;
					}

					// If the fragment nearly faces the light source (fudge)
					else
					{
						if ( diffuseLevel > -specularRollOff )
						{
							float overlightingFactor = ( specularRollOff + diffuseLevel ) / specularRollOff;

							// Calculate specular colour and add to diffuse
							accumulatedColour += specularTexColour * CalculateSpecularLighting( normal, lightVector, eye, specularPower ) * lightColour * lightPower * globalSpecularFactor * attenuation * overlightingFactor;
						}
					}
				}
			}
		}

#else
		// For each lightsource
		for ( int index = 0; index < MAX_LIGHTS; index++ )
		{
			// Early-out if the power is 0
			if ( lightType[ index ] != LIGHTTYPE_NONE )
			{
				// Calculate light direction
				// TODO: change the lighting interface so these are in object coordinates already
				float3 lightDir = mul( float4( lightDirection[ index ], 0.0f ), worldI ).xyz;

				// Calculate light attenuation
				float attenuation = CalculateLightAttenuation( _input.lightVector[ index ],
																											lightDir,
																											lightAttenuation[ index ],
																											lightType[ index ],
																											lightAngle[ index ] );

				if ( attenuation > 0.0f )
				{
					// Calculate diffuse colour and add to accumulator
					float3 lightVector = normalize( _input.lightVector[ index ] );

					float diffuseLevel = CalculateDiffuseLighting( normal, lightVector );
					accumulatedColour += diffuseTexColour * max( diffuseLevel, 0.0f ) * lightColour[ index ] * lightIntensity[ index ] * attenuation;

					// If some specular lighting should be applied
					float specularPower = lerp( minSpecPower, maxSpecPower, specularTexColour.a );
					if ( ( specularPower > 0.0f ) && ( globalSpecularFactor > 0.0f ) )
					{
						// If the fragment faces the light source
						if ( diffuseLevel > 0.0f )
						{
							// Calculate specular colour and add to diffuse
							accumulatedColour += specularTexColour * CalculateSpecularLighting( normal, lightVector, eye, specularPower ) * lightColour[ index ] * lightIntensity[ index ] * globalSpecularFactor * attenuation;
						}

						// If the fragment nearly faces the light source (fudge)
						else
						{
							if ( diffuseLevel > -specularRollOff )
							{
								float overlightingFactor = ( specularRollOff + diffuseLevel ) / specularRollOff;

								// Calculate specular colour and add to diffuse
								accumulatedColour += specularTexColour * CalculateSpecularLighting( normal, lightVector, eye, specularPower ) * lightColour[ index ] * lightIntensity[ index ] * globalSpecularFactor * attenuation * overlightingFactor;
							}
						}
					}
				}
			}
		}
#endif

		accumulatedColour.w = diffuseTexColour.w;
		_output.Colour = accumulatedColour;
	}
	else
	{
		if ( showDepth )
		{
			float depth = _input.depth.z / depthScale;
			_output.Colour = float4( depth, depth, depth, 1.0f );
		}
		else
		{
			if ( showVertexColour )
			{
				_output.Colour = _input.colour;
			}

			else
			{
				if ( showNormal )
				{
					_output.Colour = float4( normalize( _input.normal ), 1.0f );
				}
				else
				{
					if ( showTangent )
					{
						_output.Colour = float4( normalize( _input.tangent ), 1.0f );
					}
					else
					{
						if ( showBinormal )
						{
							_output.Colour = float4( normalize( _input.binormal ), 1.0f );
						}
						else
						{
							if ( showAlpha )
							{
								_output.Colour = float4( _input.colour.a, _input.colour.a, _input.colour.a, 1.0f );
							}
							else
							{
								if ( showUVMapping )
								{
									_output.Colour = float4( _input.texCoord.x, -_input.texCoord.y, 0.0f, 1.0f );
								}

								else
								{
									if ( showSpecularPower )
									{
										_output.Colour = float4( specularTexColour.a, specularTexColour.a, specularTexColour.a, 1.0f );
									}
									else
									{
										if ( showNormalLength )
										{
											float normLength = length( _input.normal );
											_output.Colour = float4( normLength, normLength, normLength, 1.0f );
										}
										else
										{
											if ( showTangentLength )
											{
												float tangentLength = length( _input.tangent );
												_output.Colour = float4( tangentLength, tangentLength, tangentLength, 1.0f );
											}
											else
											{
												if ( showBinormalLength )
												{
													float binormLength = length( _input.binormal );
													_output.Colour = float4( binormLength, binormLength, binormLength, 1.0f );
												}
												else
												{
													if ( showUVGradients )
													{
														float2 d = abs( ddx( _input.texCoord ) );
														_output.Colour = float4( d.x * 100.0f, d.y * 100.0f, 0.0f, 1.0f );
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}

	return _output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Debug
<
	bool supportsSpecialisedLighting = false;
	bool preservesGlobalState = false;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Debug";
	int    normalDeferredID		= 0;
>
{
	pass Pass0
	{
		ZEnable = 1;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx DebugVertexShader();
		PixelShader = compile sce_fp_rsx DebugFragmentShader();
#else		
		VertexShader = compile vs_3_0 DebugVertexShader();
		PixelShader = compile ps_3_0 DebugFragmentShader();
#endif		
	}
}
